#!/usr/bin/env python3
# pcsc_scardcontrol_wink.py
# Connects EXCLUSIVE + DIRECT to a SpringCard reader (even if no card is present) and sends an SCardControl
# Payload of the control is [0x58, 0x9F], i.e. the WINK command
# See https://docs.springcard.com/books/SpringCore/Host_Protocols/Direct_Protocol/CONTROL_class/IOs_and_UI/WINK

from smartcard.scard import *
import sys

def fail_and_exit(msg, rc=None):
    if rc is not None and rc != SCARD_S_SUCCESS:
        err_hex = f"0x{rc & 0xFFFFFFFF:08X}" # PC/SC error codes are known by their hexadecimal value
        err_dec = f"{rc}"                    # System-generic error codes are generally known by their decimal value
        try:
            err = f"{SCardGetErrorMessage(rc)} ({err_hex} / {err_dec})"
        except Exception:
            err = f"{err_hex} / {err_dec}"
        print(f"{msg}: {err}")
    else:
        print(msg)
    sys.exit(1)
    
def main():
    # Establish PC/SC context
    rc, hContext = SCardEstablishContext(SCARD_SCOPE_SYSTEM)
    if rc != SCARD_S_SUCCESS:
        fail_and_exit("Failed to establish context:", rc)

    # List available readers
    rc, readers = SCardListReaders(hContext, [])
    if rc != SCARD_S_SUCCESS or not readers:
        SCardReleaseContext(hContext)
        fail_and_exit("SCardListReaders failed", rc)
    if not readers:
        SCardReleaseContext(hContext)
        fail_and_exit("No PC/SC readers found")

    # Select first reader whose name starts with "SpringCard "
    reader = None
    for r in readers:
        if r.startswith("SpringCard "):
            reader = r
            break

    if not reader:
        SCardReleaseContext(hContext)
        fail_and_exit("No reader starting with 'SpringCard ' found")

    print("Using reader:", reader)

    # Connect to the reader
    rc, hCard, dwActiveProtocol = SCardConnect(
        hContext,
        reader,
        SCARD_SHARE_DIRECT,   # DIRECT access to the reader
        0                     # protocol must be 0 for DIRECT
    )
    if rc != SCARD_S_SUCCESS:
        SCardReleaseContext(hContext)
        fail_and_exit("SCardConnect (DIRECT) failed", rc)

    # Compute control code (different between Windows' PC/SC and PC/SC-Lite on Linux/macOS)
    if sys.platform.startswith("win"):
        control_code = SCARD_CTL_CODE(3500)
    else:
        control_code = SCARD_CTL_CODE(1)

    # This is the WINK command
    send_buffer = [0x58, 0x9F]

    # Perform control command
    rc, response = SCardControl(hCard, control_code, send_buffer)
    if rc == SCARD_S_SUCCESS:
        print("SCardControl OK. Response bytes:", response)
    else:
        SCardReleaseContext(hContext)
        fail_and_exit("SCardControl failed", rc)

    # Clean up the connection
    # For DIRECT, LEAVE_CARD is fine (there's no card session anyway).
    SCardDisconnect(hCard, SCARD_LEAVE_CARD)
    SCardReleaseContext(hContext)

if __name__ == "__main__":
    main()
